const Database = require('better-sqlite3');
const path = require('path');
const fs = require('fs');
const config = require('../config/config');

class DatabaseManager {
  constructor() {
    this.db = null;
  }

  /**
   * Initialize the database
   */
  initialize() {
    try {
      // Create directory if it doesn't exist
      const dbDir = path.dirname(config.database.path);
      if (!fs.existsSync(dbDir)) {
        fs.mkdirSync(dbDir, { recursive: true });
      }

      this.db = new Database(config.database.path);
      this.db.pragma('journal_mode = WAL');
      
      this.createTables();
      console.log('✅ Database initialized successfully');
    } catch (error) {
      console.error('❌ Error initializing database:', error);
      throw error;
    }
  }

  /**
   * Create necessary tables
   */
  createTables() {
    // Clients table
    this.db.exec(`
      CREATE TABLE IF NOT EXISTS clients (
        phone TEXT PRIMARY KEY,
        name TEXT,
        email TEXT,
        client_id TEXT,
        language TEXT DEFAULT 'en',
        language_confirmed INTEGER DEFAULT 0,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
      )
    `);

    // Tickets table
    this.db.exec(`
      CREATE TABLE IF NOT EXISTS tickets (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        phone TEXT NOT NULL,
        client_name TEXT,
        status TEXT DEFAULT 'open',
        priority TEXT DEFAULT 'normal',
        agent_name TEXT,
        agent_number TEXT,
        language TEXT DEFAULT 'en',
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        closed_at DATETIME,
        closed_by TEXT,
        FOREIGN KEY (phone) REFERENCES clients(phone)
      )
    `);

    // Agents table
    this.db.exec(`
      CREATE TABLE IF NOT EXISTS agents (
        phone TEXT PRIMARY KEY,
        name TEXT NOT NULL,
        language TEXT DEFAULT 'en',
        use_real_name INTEGER DEFAULT 1,
        status TEXT DEFAULT 'active',
        total_tickets INTEGER DEFAULT 0,
        tickets_today INTEGER DEFAULT 0,
        tickets_week INTEGER DEFAULT 0,
        tickets_month INTEGER DEFAULT 0,
        last_reset_day DATE,
        last_reset_week DATE,
        last_reset_month DATE,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        created_by TEXT,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
      )
    `);

    // Migration: Add status column if it doesn't exist
    try {
      const agentTableInfo = this.db.prepare("PRAGMA table_info(agents)").all();
      const hasStatus = agentTableInfo.some(col => col.name === 'status');
      
      if (!hasStatus) {
        console.log('📊 Adding status column to agents table...');
        this.db.exec('ALTER TABLE agents ADD COLUMN status TEXT DEFAULT "active"');
        console.log('✅ Column status added successfully');
      }
    } catch (error) {
      console.error('⚠️ Error adding status column:', error.message);
    }

    // Migration: Add use_real_name column if it doesn't exist
    try {
      const agentTableInfo = this.db.prepare("PRAGMA table_info(agents)").all();
      const hasUseRealName = agentTableInfo.some(col => col.name === 'use_real_name');
      
      if (!hasUseRealName) {
        console.log('📊 Adding use_real_name column to agents table...');
        this.db.exec('ALTER TABLE agents ADD COLUMN use_real_name INTEGER DEFAULT 1');
        console.log('✅ Column use_real_name added successfully');
      }
    } catch (error) {
      console.error('⚠️ Error adding use_real_name column:', error.message);
    }

    // Migration: Add statistics columns if they don't exist
    try {
      const agentTableInfo = this.db.prepare("PRAGMA table_info(agents)").all();
      const columnNames = agentTableInfo.map(col => col.name);
      
      const statsColumns = [
        { name: 'total_tickets', type: 'INTEGER DEFAULT 0' },
        { name: 'tickets_today', type: 'INTEGER DEFAULT 0' },
        { name: 'tickets_week', type: 'INTEGER DEFAULT 0' },
        { name: 'tickets_month', type: 'INTEGER DEFAULT 0' },
        { name: 'last_reset_day', type: 'DATE' },
        { name: 'last_reset_week', type: 'DATE' },
        { name: 'last_reset_month', type: 'DATE' },
        { name: 'created_by', type: 'TEXT' },
        { name: 'updated_at', type: 'DATETIME DEFAULT CURRENT_TIMESTAMP' }
      ];
      
      for (const col of statsColumns) {
        if (!columnNames.includes(col.name)) {
          console.log(`📊 Adding ${col.name} column to agents table...`);
          this.db.exec(`ALTER TABLE agents ADD COLUMN ${col.name} ${col.type}`);
          console.log(`✅ Column ${col.name} added successfully`);
        }
      }
    } catch (error) {
      console.error('⚠️ Error adding statistics columns:', error.message);
    }

    // Migration: Add closed_by column if it doesn't exist
    try {
      const tableInfo = this.db.prepare("PRAGMA table_info(tickets)").all();
      const hasClosedBy = tableInfo.some(col => col.name === 'closed_by');
      
      if (!hasClosedBy) {
        console.log('📊 Adding closed_by column to tickets table...');
        this.db.exec('ALTER TABLE tickets ADD COLUMN closed_by TEXT');
        console.log('✅ Column closed_by added successfully');
      }
    } catch (error) {
      console.error('⚠️ Error adding closed_by column:', error.message);
    }

    // Messages table
    this.db.exec(`
      CREATE TABLE IF NOT EXISTS messages (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        ticket_id INTEGER NOT NULL,
        phone TEXT NOT NULL,
        sender_type TEXT NOT NULL,
        message_type TEXT DEFAULT 'text',
        content TEXT,
        translated_content TEXT,
        media_url TEXT,
        original_language TEXT,
        target_language TEXT,
        delivered INTEGER DEFAULT 0,
        read INTEGER DEFAULT 0,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (ticket_id) REFERENCES tickets(id)
      )
    `);

    // Tabela de notificações
    this.db.exec(`
      CREATE TABLE IF NOT EXISTS notifications (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        ticket_id INTEGER NOT NULL,
        notification_type TEXT NOT NULL,
        recipient TEXT NOT NULL,
        message TEXT,
        sent INTEGER DEFAULT 0,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        sent_at DATETIME,
        FOREIGN KEY (ticket_id) REFERENCES tickets(id)
      )
    `);

    // Tabela de configurações
    this.db.exec(`
      CREATE TABLE IF NOT EXISTS settings (
        key TEXT PRIMARY KEY,
        value TEXT,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
      )
    `);

    // Departments table
    this.db.exec(`
      CREATE TABLE IF NOT EXISTS departments (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        name TEXT NOT NULL UNIQUE,
        description TEXT,
        is_general INTEGER DEFAULT 0,
        status TEXT DEFAULT 'active',
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
      )
    `);

    // Agent-Department relationship table
    this.db.exec(`
      CREATE TABLE IF NOT EXISTS agent_departments (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        agent_phone TEXT NOT NULL,
        department_id INTEGER NOT NULL,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (agent_phone) REFERENCES agents(phone) ON DELETE CASCADE,
        FOREIGN KEY (department_id) REFERENCES departments(id) ON DELETE CASCADE,
        UNIQUE(agent_phone, department_id)
      )
    `);

    // Add status column to departments table if it doesn't exist
    try {
      const deptTableInfo = this.db.prepare("PRAGMA table_info(departments)").all();
      const hasStatus = deptTableInfo.some(col => col.name === 'status');
      
      if (!hasStatus) {
        console.log('📊 Adding status column to departments table...');
        this.db.exec('ALTER TABLE departments ADD COLUMN status TEXT DEFAULT "active"');
        console.log('✅ Status column added successfully');
      }
    } catch (error) {
      console.error('⚠️ Error adding status column to departments:', error.message);
    }

    // Add department_id to tickets table
    try {
      const ticketTableInfo = this.db.prepare("PRAGMA table_info(tickets)").all();
      const hasDepartment = ticketTableInfo.some(col => col.name === 'department_id');
      
      if (!hasDepartment) {
        console.log('📊 Adding department_id column to tickets table...');
        this.db.exec('ALTER TABLE tickets ADD COLUMN department_id INTEGER');
        this.db.exec('ALTER TABLE tickets ADD COLUMN department_name TEXT');
        console.log('✅ Department columns added successfully');
      }
    } catch (error) {
      console.error('⚠️ Error adding department columns:', error.message);
    }

    // Índices para melhor performance
    this.db.exec(`
      CREATE INDEX IF NOT EXISTS idx_tickets_phone ON tickets(phone);
      CREATE INDEX IF NOT EXISTS idx_tickets_status ON tickets(status);
      CREATE INDEX IF NOT EXISTS idx_messages_ticket ON messages(ticket_id);
      CREATE INDEX IF NOT EXISTS idx_notifications_ticket ON notifications(ticket_id);
    `);
  }

  /**
   * Salva ou atualiza cliente
   */
  saveClient(phone, data) {
    // Aceita 'account' como alias para 'email'
    const email = data.email || data.account || null;
    
    const stmt = this.db.prepare(`
      INSERT INTO clients (phone, name, email, client_id, language, language_confirmed)
      VALUES (?, ?, ?, ?, ?, ?)
      ON CONFLICT(phone) DO UPDATE SET
        name = COALESCE(excluded.name, name),
        email = COALESCE(excluded.email, email),
        client_id = COALESCE(excluded.client_id, client_id),
        language = COALESCE(excluded.language, language),
        language_confirmed = COALESCE(excluded.language_confirmed, language_confirmed),
        updated_at = CURRENT_TIMESTAMP
    `);

    return stmt.run(
      phone,
      data.name || null,
      email,
      data.client_id || null,
      data.language || 'en',
      data.language_confirmed || 0
    );
  }

  /**
   * Obtém cliente por telefone
   */
  getClient(phone) {
    const stmt = this.db.prepare('SELECT * FROM clients WHERE phone = ?');
    const client = stmt.get(phone);
    
    // Adiciona 'account' como alias para 'email' (compatibilidade)
    if (client) {
      client.account = client.email;
    }
    
    return client;
  }

  /**
   * Cria novo ticket
   */
  createTicket(phone, clientName, language, departmentId = null) {
    const stmt = this.db.prepare(`
      INSERT INTO tickets (phone, client_name, language, status, department_id)
      VALUES (?, ?, ?, 'open', ?)
    `);

    const result = stmt.run(phone, clientName, language, departmentId);
    return result.lastInsertRowid;
  }

  /**
   * Obtém ticket aberto do cliente
   */
  getOpenTicket(phone) {
    const stmt = this.db.prepare(`
      SELECT * FROM tickets 
      WHERE phone = ? AND status IN ('open', 'assigned', 'transferring')
      ORDER BY created_at DESC
      LIMIT 1
    `);
    return stmt.get(phone);
  }

  /**
   * Obtém ticket por ID
   */
  getTicket(ticketId) {
    const stmt = this.db.prepare('SELECT * FROM tickets WHERE id = ?');
    return stmt.get(ticketId);
  }

  /**
   * Lista todos os tickets abertos
   */
  listOpenTickets() {
    const stmt = this.db.prepare(`
      SELECT * FROM tickets 
      WHERE status IN ('open', 'assigned', 'transferring')
      ORDER BY created_at DESC
    `);
    return stmt.all();
  }

  /**
   * Lista tickets fechados
   */
  listClosedTickets(limit = 20) {
    const stmt = this.db.prepare(`
      SELECT * FROM tickets 
      WHERE status = 'closed'
      ORDER BY closed_at DESC
      LIMIT ?
    `);
    return stmt.all(limit);
  }

  /**
   * Obtém ticket aberto que está sendo atendido por um agente específico
   */
  getOpenTicketByAgent(agentPhone) {
    const stmt = this.db.prepare(`
      SELECT * FROM tickets 
      WHERE agent_number = ? AND (status = 'assigned' OR status = 'open')
      ORDER BY updated_at DESC
      LIMIT 1
    `);
    return stmt.get(agentPhone);
  }

  /**
   * Atualiza ticket
   */
  updateTicket(ticketId, data) {
    const fields = [];
    const values = [];

    if (data.status !== undefined) {
      fields.push('status = ?');
      values.push(data.status);
    }
    if (data.agent_name !== undefined) {
      fields.push('agent_name = ?');
      values.push(data.agent_name);
    }
    if (data.agent_number !== undefined) {
      fields.push('agent_number = ?');
      values.push(data.agent_number);
    }
    if (data.priority !== undefined) {
      fields.push('priority = ?');
      values.push(data.priority);
    }
    if (data.closed_by !== undefined) {
      fields.push('closed_by = ?');
      values.push(data.closed_by);
    }

    fields.push('updated_at = CURRENT_TIMESTAMP');

    if (data.status === 'closed') {
      fields.push('closed_at = CURRENT_TIMESTAMP');
    }

    values.push(ticketId);

    const stmt = this.db.prepare(`
      UPDATE tickets SET ${fields.join(', ')}
      WHERE id = ?
    `);

    return stmt.run(...values);
  }

  /**
   * Adiciona mensagem ao ticket
   */
  addMessage(ticketId, phone, senderType, content, messageType = 'text', translatedContent = null, mediaUrl = null, originalLanguage = null, targetLanguage = null) {
    const stmt = this.db.prepare(`
      INSERT INTO messages 
      (ticket_id, phone, sender_type, message_type, content, translated_content, media_url, original_language, target_language)
      VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
    `);

    const result = stmt.run(
      ticketId,
      phone,
      senderType,
      messageType,
      content,
      translatedContent,
      mediaUrl,
      originalLanguage,
      targetLanguage
    );

    return result.lastInsertRowid;
  }

  /**
   * Obtém mensagens de um ticket
   */
  getTicketMessages(ticketId) {
    const stmt = this.db.prepare(`
      SELECT * FROM messages 
      WHERE ticket_id = ?
      ORDER BY created_at ASC
    `);
    return stmt.all(ticketId);
  }

  /**
   * Cria notificação
   */
  createNotification(ticketId, type, recipient, message) {
    const stmt = this.db.prepare(`
      INSERT INTO notifications (ticket_id, notification_type, recipient, message)
      VALUES (?, ?, ?, ?)
    `);

    const result = stmt.run(ticketId, type, recipient, message);
    return result.lastInsertRowid;
  }

  /**
   * Marca notificação como enviada
   */
  markNotificationSent(notificationId) {
    const stmt = this.db.prepare(`
      UPDATE notifications 
      SET sent = 1, sent_at = CURRENT_TIMESTAMP
      WHERE id = ?
    `);
    return stmt.run(notificationId);
  }

  /**
   * Obtém notificações pendentes
   */
  getPendingNotifications() {
    const stmt = this.db.prepare(`
      SELECT n.*, t.phone, t.client_name
      FROM notifications n
      JOIN tickets t ON n.ticket_id = t.id
      WHERE n.sent = 0
      ORDER BY n.created_at ASC
    `);
    return stmt.all();
  }

  /**
   * Obtém estatísticas gerais
   */
  getStats() {
    try {
      const openTickets = this.db.prepare("SELECT COUNT(*) as count FROM tickets WHERE status IN ('open', 'assigned', 'transferring')").get();
      const closedTickets = this.db.prepare("SELECT COUNT(*) as count FROM tickets WHERE status = 'closed'").get();
      const totalClients = this.db.prepare('SELECT COUNT(*) as count FROM clients').get();
      const totalMessages = this.db.prepare('SELECT COUNT(*) as count FROM messages').get();

      return {
        openTickets: openTickets.count,
        closedTickets: closedTickets.count,
        totalClients: totalClients.count,
        totalMessages: totalMessages.count
      };
    } catch (error) {
      console.error('❌ Erro ao obter estatísticas:', error.message);
      return {
        openTickets: 0,
        closedTickets: 0,
        totalClients: 0,
        totalMessages: 0
      };
    }
  }

  // ==================== AGENT MANAGEMENT ====================

  /**
   * Add new agent
   */
  addAgent(phone, name, language, createdBy) {
    const stmt = this.db.prepare(`
      INSERT INTO agents (phone, name, language, created_by, last_reset_day, last_reset_week, last_reset_month)
      VALUES (?, ?, ?, ?, DATE('now'), DATE('now'), DATE('now'))
    `);
    return stmt.run(phone, name, language, createdBy);
  }

  /**
   * Get agent by phone
   */
  getAgent(phone) {
    const stmt = this.db.prepare('SELECT * FROM agents WHERE phone = ?');
    return stmt.get(phone);
  }

  /**
   * Get all agents
   */
  getAllAgents() {
    try {
      // Try with status column first
      const stmt = this.db.prepare("SELECT * FROM agents WHERE status = 'active' ORDER BY name");
      return stmt.all();
    } catch (error) {
      // If status column doesn't exist, get all agents
      console.log('⚠️ Status column not found, returning all agents');
      const stmt = this.db.prepare('SELECT * FROM agents ORDER BY name');
      return stmt.all();
    }
  }

  /**
   * Update agent
   */
  updateAgent(phone, data) {
    const fields = [];
    const values = [];

    if (data.name !== undefined) {
      fields.push('name = ?');
      values.push(data.name);
    }

    if (data.language !== undefined) {
      fields.push('language = ?');
      values.push(data.language);
    }

    if (data.status !== undefined) {
      fields.push('status = ?');
      values.push(data.status);
    }

    if (fields.length === 0) {
      return;
    }

    fields.push('updated_at = CURRENT_TIMESTAMP');
    values.push(phone);

    const sql = `UPDATE agents SET ${fields.join(', ')} WHERE phone = ?`;
    const stmt = this.db.prepare(sql);
    return stmt.run(...values);
  }

  /**
   * Remove agent (set inactive or delete)
   */
  removeAgent(phone) {
    try {
      // Try to set status to inactive if column exists
      const stmt = this.db.prepare('UPDATE agents SET status = "inactive", updated_at = CURRENT_TIMESTAMP WHERE phone = ?');
      return stmt.run(phone);
    } catch (error) {
      // If status column doesn't exist, delete the agent
      console.log('⚠️ Status column not found, deleting agent instead');
      const stmt = this.db.prepare('DELETE FROM agents WHERE phone = ?');
      return stmt.run(phone);
    }
  }

  /**
   * Increment agent ticket statistics
   */
  incrementAgentStats(phone) {
    // Reset statistics if needed
    const agent = this.getAgent(phone);
    if (!agent) return;

    const today = new Date().toISOString().split('T')[0];
    const updates = ['total_tickets = total_tickets + 1'];
    
    // Reset daily stats if new day
    if (agent.last_reset_day !== today) {
      updates.push('tickets_today = 1');
      updates.push(`last_reset_day = '${today}'`);
    } else {
      updates.push('tickets_today = tickets_today + 1');
    }

    // Reset weekly stats if new week
    const weekStart = this.getWeekStart();
    if (agent.last_reset_week !== weekStart) {
      updates.push('tickets_week = 1');
      updates.push(`last_reset_week = '${weekStart}'`);
    } else {
      updates.push('tickets_week = tickets_week + 1');
    }

    // Reset monthly stats if new month
    const monthStart = this.getMonthStart();
    if (agent.last_reset_month !== monthStart) {
      updates.push('tickets_month = 1');
      updates.push(`last_reset_month = '${monthStart}'`);
    } else {
      updates.push('tickets_month = tickets_month + 1');
    }

    const stmt = this.db.prepare(`
      UPDATE agents SET ${updates.join(', ')}, updated_at = CURRENT_TIMESTAMP
      WHERE phone = ?
    `);
    return stmt.run(phone);
  }

  /**
   * Get week start date (Monday)
   */
  getWeekStart() {
    const now = new Date();
    const day = now.getDay();
    const diff = now.getDate() - day + (day === 0 ? -6 : 1); // adjust when day is sunday
    const monday = new Date(now.setDate(diff));
    return monday.toISOString().split('T')[0];
  }

  /**
   * Get month start date
   */
  getMonthStart() {
    const now = new Date();
    return new Date(now.getFullYear(), now.getMonth(), 1).toISOString().split('T')[0];
  }

  /**
   * Get agent statistics
   */
  getAgentStats(phone) {
    const agent = this.getAgent(phone);
    if (!agent) return null;

    return {
      name: agent.name,
      phone: agent.phone,
      language: agent.language,
      total: agent.total_tickets,
      today: agent.tickets_today,
      week: agent.tickets_week,
      month: agent.tickets_month,
      status: agent.status
    };
  }

  /**
   * Get all agents statistics
   */
  getAllAgentsStats() {
    const agents = this.getAllAgents();
    return agents.map(agent => ({
      name: agent.name,
      phone: agent.phone,
      language: agent.language,
      total: agent.total_tickets,
      today: agent.tickets_today,
      week: agent.tickets_week,
      month: agent.tickets_month
    }));
  }

  // ==================== SETTINGS MANAGEMENT ====================

  /**
   * Save a setting to database
   * @param {string} key - Setting key
   * @param {string} value - Setting value
   */
  saveSetting(key, value) {
    const stmt = this.db.prepare(`
      INSERT INTO settings (key, value, updated_at)
      VALUES (?, ?, CURRENT_TIMESTAMP)
      ON CONFLICT(key) DO UPDATE SET 
        value = excluded.value,
        updated_at = CURRENT_TIMESTAMP
    `);
    return stmt.run(key, value);
  }

  /**
   * Get a setting from database
   * @param {string} key - Setting key
   * @returns {string|null} - Setting value or null
   */
  getSetting(key) {
    try {
      const stmt = this.db.prepare('SELECT value FROM settings WHERE key = ?');
      const row = stmt.get(key);
      return row ? row.value : null;
    } catch (error) {
      console.error(`Error getting setting ${key}:`, error);
      return null;
    }
  }

  /**
   * Set a setting in database
   * @param {string} key - Setting key
   * @param {string} value - Setting value
   * @returns {boolean} - Success status
   */
  setSetting(key, value) {
    try {
      const stmt = this.db.prepare(`
        INSERT INTO settings (key, value)
        VALUES (?, ?)
        ON CONFLICT(key) DO UPDATE SET value = excluded.value, updated_at = CURRENT_TIMESTAMP
      `);
      stmt.run(key, String(value));
      return true;
    } catch (error) {
      console.error(`Error setting ${key}:`, error);
      return false;
    }
  }

  /**
   * Get all settings
   * @returns {Object} - All settings as key-value pairs
   */
  getAllSettings() {
    const stmt = this.db.prepare('SELECT key, value FROM settings');
    const rows = stmt.all();
    const settings = {};
    for (const row of rows) {
      settings[row.key] = row.value;
    }
    return settings;
  }

  /**
   * Delete a setting
   * @param {string} key - Setting key
   */
  deleteSetting(key) {
    const stmt = this.db.prepare('DELETE FROM settings WHERE key = ?');
    return stmt.run(key);
  }

  // ==================== DEPARTMENT METHODS ====================

  /**
   * Create a new department
   */
  createDepartment(name, description = '', isGeneral = 0) {
    try {
      const stmt = this.db.prepare(`
        INSERT INTO departments (name, description, is_general, status)
        VALUES (?, ?, ?, 'active')
      `);
      const result = stmt.run(name, description, isGeneral);
      return result.lastInsertRowid;
    } catch (error) {
      console.error('Error creating department:', error);
      return null;
    }
  }

  /**
   * Get all departments
   */
  listDepartments(activeOnly = false) {
    try {
      let query = 'SELECT * FROM departments';
      if (activeOnly) {
        query += " WHERE status = 'active'";
      }
      query += ' ORDER BY is_general DESC, name ASC';
      
      const stmt = this.db.prepare(query);
      return stmt.all();
    } catch (error) {
      console.error('Error listing departments:', error);
      return [];
    }
  }

  /**
   * Get department by ID
   */
  getDepartment(id) {
    try {
      const stmt = this.db.prepare('SELECT * FROM departments WHERE id = ?');
      return stmt.get(id);
    } catch (error) {
      console.error('Error getting department:', error);
      return null;
    }
  }

  /**
   * Update department
   */
  updateDepartment(id, data) {
    try {
      const fields = [];
      const values = [];

      if (data.name !== undefined) {
        fields.push('name = ?');
        values.push(data.name);
      }
      if (data.description !== undefined) {
        fields.push('description = ?');
        values.push(data.description);
      }
      if (data.status !== undefined) {
        fields.push('status = ?');
        values.push(data.status);
      }
      if (data.is_general !== undefined) {
        fields.push('is_general = ?');
        values.push(data.is_general);
      }

      fields.push('updated_at = CURRENT_TIMESTAMP');
      values.push(id);

      const stmt = this.db.prepare(`
        UPDATE departments 
        SET ${fields.join(', ')}
        WHERE id = ?
      `);
      stmt.run(...values);
      return true;
    } catch (error) {
      console.error('Error updating department:', error);
      return false;
    }
  }

  /**
   * Delete department
   */
  deleteDepartment(id) {
    try {
      const stmt = this.db.prepare('DELETE FROM departments WHERE id = ?');
      stmt.run(id);
      return true;
    } catch (error) {
      console.error('Error deleting department:', error);
      return false;
    }
  }

  /**
   * Add agent to department
   */
  addAgentToDepartment(agentPhone, departmentId) {
    try {
      const stmt = this.db.prepare(`
        INSERT INTO agent_departments (agent_phone, department_id)
        VALUES (?, ?)
      `);
      stmt.run(agentPhone, departmentId);
      return true;
    } catch (error) {
      if (error.message.includes('UNIQUE')) {
        return false; // Already exists
      }
      console.error('Error adding agent to department:', error);
      return false;
    }
  }

  /**
   * Remove agent from department
   */
  removeAgentFromDepartment(agentPhone, departmentId) {
    try {
      const stmt = this.db.prepare(`
        DELETE FROM agent_departments 
        WHERE agent_phone = ? AND department_id = ?
      `);
      stmt.run(agentPhone, departmentId);
      return true;
    } catch (error) {
      console.error('Error removing agent from department:', error);
      return false;
    }
  }

  /**
   * Get agents in a department
   */
  getDepartmentAgents(departmentId) {
    try {
      const stmt = this.db.prepare(`
        SELECT a.* 
        FROM agents a
        INNER JOIN agent_departments ad ON a.phone = ad.agent_phone
        WHERE ad.department_id = ? AND a.status = 'active'
        ORDER BY a.name
      `);
      return stmt.all(departmentId);
    } catch (error) {
      console.error('Error getting department agents:', error);
      return [];
    }
  }

  /**
   * Get agent's departments
   */
  getAgentDepartments(agentPhone) {
    try {
      const stmt = this.db.prepare(`
        SELECT d.* 
        FROM departments d
        INNER JOIN agent_departments ad ON d.id = ad.department_id
        WHERE ad.agent_phone = ? AND d.status = 'active'
        ORDER BY d.is_general DESC, d.name
      `);
      return stmt.all(agentPhone);
    } catch (error) {
      console.error('Error getting agent departments:', error);
      return [];
    }
  }

  /**
   * Get general department (receives all notifications)
   */
  getGeneralDepartment() {
    try {
      const stmt = this.db.prepare(`
        SELECT * FROM departments 
        WHERE is_general = 1 AND status = 'active'
        LIMIT 1
      `);
      return stmt.get();
    } catch (error) {
      console.error('Error getting general department:', error);
      return null;
    }
  }

  /**
   * Check if departments system is enabled
   */
  isDepartmentsEnabled() {
    const enabled = this.getSetting('departments_enabled');
    return enabled === '1';
  }

  /**
   * Fecha conexão com banco de dados
   */
  close() {
    if (this.db) {
      this.db.close();
      console.log('✅ Banco de dados fechado');
    }
  }
}

module.exports = new DatabaseManager();
